//
// Arduino IDE による KOROBO2 / KOROBO3 の開発サンプル
//   モータードライバーシールド SU-1201 とは，ピンアサインが異なっています．
//
// HIGH 100 ～ 180
// MID   60 ～ 100
// LOW   20 ～  60

#define LED4 4
#define BZ   11
#define MT_R1  6  // 0 ～ 255
#define MT_R2  5  // 0 ～ 255
#define MT_L1 10  // 0 ～ 255
#define MT_L2  9  // 0 ～ 255
#define SW_ST 13
#define SW_BTN 12
#define PSD_L_RECV 8
#define PSD_L_TRANS 14
#define PSD_R_RECV 7
#define PSD_R_TRANS 16
#define TOUCH_L 18
#define TOUCH_R 19
#define CH4_3   21
#define CH4_4   22
#define CH4_5   23
#define CH4_6   24
#define CH4_7   25

/*
 * 書式 krb3_move( 速度v, 角速度a)
 * 
 * 前進:      0 < v < 120
 * STOP:         v = 0
 * バック: -120 < v <   0
 * 
 * 直進:           a = 0
 * 左旋回:    0 <  a <= 180
 * 右旋回: -180 <= a <   0
 * 
 * Example:
 *   直進     krb3_move( 速度, 0)
 *   左前進   krb3_move( 速度, 角速度)   0 < 角速度 <= 90
 *   右前進   krb3_move( 速度, 角速度) -90 <= 角速度 < 0
 *   バック   krb3_move( -速度, 0)
 *   左スピン krb3_move( 速度, 180)
 *   右スピン krb3_move( 速度, -180)
 *
 *   v*( a/RIGHT_ANG)の計算に map( a, 0, 90, 0, v_max) 0 <= a <= 90 を使う!?
 */
#define RIGHT_ANG 90         // 直角 90 deg

int krb3_move( int v, int a){
  if( a >= -RIGHT_ANG || a <= RIGHT_ANG){ // 前進，左右前進，後退，左右後退 → 左右信地旋回
    if( v>0){                       // KOROBO3 前進
      analogWrite( MT_R1, v + v*( a/RIGHT_ANG));   // PWM 値を最大/2に設定 右モーター正転 （前進）
      analogWrite( MT_R2, 0);
      analogWrite( MT_L1, 0);   // PWM 値を最大/2に設定　左モーター逆転　（前進）
      analogWrite( MT_L2, v - v*( a/RIGHT_ANG));
    }
    else if( v<0){                  // KOROBO3 バック
      analogWrite( MT_R1, 0);   // PWM 値を最大/2に設定 右モーター逆転 （バック）
      analogWrite( MT_R2, -v - v*( a/RIGHT_ANG));
      analogWrite( MT_L1, -v + v*( a/RIGHT_ANG));   // PWM 値を最大/2に設定　左モーター正転　（バック）
      analogWrite( MT_L2, 0);
    }
    else{
      analogWrite( MT_R1, 0);     // 0 でブレーキ
      analogWrite( MT_R2, 0);     // 0 でブレーキ
      analogWrite( MT_L1, 0);     // 0 でブレーキ
      analogWrite( MT_L2, 0);     // 0 でブレーキ
    }
  }
  else{
    if( v>0){                       // KOROBO3 左右前進信地旋回 → 超信地旋回
      analogWrite( MT_R1, v + v*( ( a-RIGHT_ANG)/RIGHT_ANG));   // PWM 値を最大/2に設定 右モーター正転 （前進）
      analogWrite( MT_R2, 0);
      analogWrite( MT_L1, v - v*( ( a-RIGHT_ANG)/RIGHT_ANG));   // PWM 値を最大/2に設定　左モーター逆転　（前進）
      analogWrite( MT_L2, 0);
    }
    else if( v<0){                  // KOROBO3 左右バック信地旋回 → 超信地旋回
      analogWrite( MT_R1, 0);   // PWM 値を最大/2に設定 右モーター逆転 （バック）
      analogWrite( MT_R2, -v - v*( ( a-RIGHT_ANG)/RIGHT_ANG));
      analogWrite( MT_L1, 0);   // PWM 値を最大/2に設定　左モーター正転　（バック）
      analogWrite( MT_L2, -v + v*( ( a-RIGHT_ANG)/RIGHT_ANG));
    }
  }
}

int krb3_pin_setup(){
  pinMode( LED4, OUTPUT);    // KOROBO2/ KOROBO3 の LED1 (8) と LED2は，光センサのON/OFF表示に予約済
  pinMode( BZ,   OUTPUT);    // ブザー用

  pinMode( MT_R1, OUTPUT);     // 右モーター用のピンの出力設定
  pinMode( MT_R2, OUTPUT);     // 右モーター用のピンの出力設定
  pinMode( MT_L1, OUTPUT);     // 左モーター用のピンの出力設定
  pinMode( MT_L2, OUTPUT);     // 左モーター用のピンの出\力設定
  pinMode( PSD_L_TRANS, OUTPUT);     // 左光センサのLED
  pinMode( PSD_R_TRANS, OUTPUT);     // 右光センサのLED

  pinMode( SW_ST,  INPUT_PULLUP);
  pinMode( SW_BTN, INPUT_PULLUP);
  pinMode( PSD_L_TRANS, INPUT);     // 左光センサ
  pinMode( PSD_R_TRANS, INPUT);     // 右光センサ

  pinMode( TOUCH_L, INPUT_PULLUP);
  pinMode( TOUCH_R, INPUT_PULLUP);

  analogWrite( PSD_L_TRANS, 127);     // 左光センサのLED点灯
  analogWrite( PSD_R_TRANS, 127);     // 右光センサのLED点灯
}

int krb3_IO_test() {
	digitalWrite( LED4, HIGH);    // LED1, LD2 で実行すると，LEDの明滅ではなく，明るさが変化する．
	delay( 500);
	digitalWrite( LED4, LOW);
	delay( 500);

	analogWrite( BZ, 128); // ブザーを鳴らす
	delay( 500);           // 0.5 秒待つ
	analogWrite( BZ, 0);   // ブザーを止める
	delay( 500);           // 0.5 秒待つ

  analogWrite( MT_R1, 127);   // PWM 値を最大/2に設定 右モーター正転 （前進）
  analogWrite( MT_R2, 0);
  delay( 500);            // 0.5秒の動作待ち時間
  analogWrite( MT_R1, 0);     // 0 でブレーキ
  analogWrite( MT_R2, 0);     // 0 でブレーキ
  delay( 500);           // 1 秒の動作待ち時間

  analogWrite( MT_R1, 0);   // PWM 値を最大/2に設定　右モーター逆転　（バック）
  analogWrite( MT_R2, 127);
  delay( 500);            // 0.5秒の動作待ち時間
  analogWrite( MT_R1, 0);     // PWM 値を 0 でブレーキ
  analogWrite( MT_R2, 0);     // PWM 値を 0 でブレーキ
  delay( 500);           // 1 秒の動作待ち時間

  analogWrite( MT_L1, 127);   // PWM 値を最大/2に設定 左モーター正転　（バック）
  analogWrite( MT_L2, 0);
  delay( 500);            // 0.5秒の動作待ち時間
  analogWrite( MT_L1, 0);     // 0 でブレーキ
  analogWrite( MT_L2, 0);     // 0 でブレーキ
  delay( 500);           // 1 秒の動作待ち時間

  analogWrite( MT_L1, 0);   // PWM 値を最大/2に設定　左モーター逆転　（前進）
  analogWrite( MT_L2, 127);
  delay( 500);            // 0.5秒の動作待ち時間
  analogWrite( MT_L1, 0);     // PWM 値を 0 でブレーキ
  analogWrite( MT_L2, 0);     // PWM 値を 0 でブレーキ
  delay( 500);           // 1 秒の動作待ち時間

  if( digitalRead( SW_ST) == LOW){   // SW_ST (青) が押されてたら
    for( int i=0; i<3; i++){
      analogWrite( BZ, 127);
      delay( 100);
      analogWrite( BZ, 0);
      delay( 100);
    }
  }

  if( digitalRead( SW_BTN) == LOW){   // SW_BTN (黄) が押されてたら
    for( int i=0; i<3; i++){
      analogWrite( BZ, 64);
      delay( 100);
      analogWrite( BZ, 0);
      delay( 100);
    }
  }

  if( digitalRead( PSD_L_RECV) == LOW){   // 左光センサの下が黒だったら
    for( int i=0; i<8; i++){
      analogWrite( BZ, 127);
      delay( 50);
      analogWrite( BZ, 0);
      delay( 50);
    }
  }
}


void setup(){
  krb3_pin_setup();
  while( digitalRead( SW_ST) == HIGH){}   // SW_ST (青) が押されるまで，停止
}

void loop(){
  krb3_IO_test();
}
